<?php

namespace Modules\Platform\User\Http\Controllers\User;

use Intervention\Image\Facades\Image;
use Illuminate\Support\Facades\Request;
use Modules\Platform\Core\Helper\UserHelper;
use Modules\Platform\User\Http\Forms\UserForm;
use Modules\Platform\User\Datatables\UsersDatatable;
use Modules\Platform\User\Http\Requests\UserRequest;
use Modules\Platform\User\Repositories\UserRepository;
use Modules\Platform\User\Http\Forms\ChangePasswordForm;
use Modules\Platform\User\Http\Requests\UserCreateRequest;
use Modules\Platform\User\Http\Requests\UserUpdateRequest;
use Modules\Platform\User\Http\Requests\AccountUpdateRequest;
use Modules\Platform\Core\Http\Controllers\SettingsCrudController;

/**  
 * CRUD UserController
 *  
 * Class UserController
 * @package Modules\Platform\User\Http\Controllers\User 
 */
class UserController extends SettingsCrudController
{

    protected $demoMode = false;

    protected $datatable = UsersDatatable::class;

    protected $formClass = UserForm::class;

    protected $storeRequest = UserCreateRequest::class;

    protected $updateRequest = UserUpdateRequest::class;

    protected $repository = UserRepository::class;

    protected $moduleName = 'user';

    protected $jsFiles = [
        'BAP_Users.js'
    ];

    protected $includeViews = [
        'user::users.changePassword'
    ];

    protected $languageFile = 'user::users';

    protected $showFields = [
     
        'login_and_role' => [
            'company_id' => [
                'type' => 'manyToOne',
                'relation' => 'Company', 
                'column' => 'name', 
                'belongs_to' => 'Company'
            ],
            'place_id' => [
                'type' => 'manyToOne',
                'relation' => 'Place',
                'column' => 'identifier_name',
                'belongs_to' => 'Place',
            ],
            'is_active' => ['type' => 'switch', 'col-class' => 'col-md-12'],
            // 'access_to_all_entity' => ['type' => 'switch', 'col-class' => 'col-md-6'],
            'first_name' => ['type' => 'text',  'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'],
            'last_name' => ['type' => 'text',  'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'],
            'email' => ['type' => 'email',  'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'],
            'roles' => ['type' => 'oneToMany', 'relation' => 'roles', 'column' => 'display_name','col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'],
        ],
        'more' => [
            'person_tax_number' => ['type' => 'text','col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'],
            'birthday' => ['type' => 'date','col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'],
            'home_phone' => ['type' => 'text','col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'],
            'mobile_phone' => ['type' => 'text','col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'],
        ],
        'address' => [
            'address_state' => ['type' => 'text','col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'],
            'address_city' => ['type' => 'text','col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'],
            'address_street' => ['type' => 'text','col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'],
            'number' => ['type' => 'text','col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'],
            'complement' => ['type' => 'text','col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'],
            'neighborhood' => ['type' => 'text','col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'],
            'address_postal_code' => ['type' => 'text','col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'],
        ],
        // 'more' => [
        //     'title' => ['type' => 'text'],
        //     'department' => ['type' => 'text'],
        //     'office_phone' => ['type' => 'text'],
        //     'mobile_phone' => ['type' => 'text'],
        //     'home_phone' => ['type' => 'text'],
        //     'person_tax_number' => ['type' => 'text'],
        //     'personal_id' => ['type' => 'text'],
        //     'birthday' => ['type' => 'date'],
        //     'secondary_email' => ['type' => 'email'],
        // ],

        'language' => [
            'language_id' => ['type' => 'manyToOne', 'relation' => 'Language', 'column' => 'name', 'belongs_to' => 'Language','col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'],
            'time_format_id' => ['type' => 'manyToOne', 'relation' => 'TimeFormat', 'column' => 'name', 'belongs_to' => 'TimeFormat','col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'],
        ],

        'profile' => [
            'profile_pic_conf' => ['type' => 'localized_text'],
            'profile_picture' => ['type' => 'file', 'hide_in_show' => false]

        ],

        'notes' => [
            'notes' => [
                'type' => 'text',
            ],                
        ],

    ];


    protected $routes = [];

    public function __construct() { 

        if(Request()->route()->getPrefix() == 'users/'){
            $this->routes = [
                'index' => 'users.index',
                'show' => 'users.show',
                'create' => 'users.create',
                'edit' => 'users.edit',
                'update' => 'users.update',
                'store' => 'users.store',
                'destroy' => 'users.destroy'
            ];
            unset($this->showFields['company']);
        }else{
            $this->routes = [
                'index' => 'settings.users.index',
                'show' => 'settings.users.show',
                'create' => 'settings.users.create',
                'edit' => 'settings.users.edit',
                'update' => 'settings.users.update',
                'store' => 'settings.users.store',
                'destroy' => 'settings.users.destroy'
            ];
        }

        parent::__construct();

    }

    /**
     * Overwritten function for add password form
     * @param $identifier
     * @return $this|\Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function show($identifier)
    {
        $view = parent::show($identifier);

        $route = '';

        if(Request()->route()->getPrefix() == 'users/'){
            $route = 'users.users.change-password';
        }else{
            $route = 'settings.users.change-password';
        }

        $passwordForm = $this->form(ChangePasswordForm::class, [
            'method' => 'POST',
            'url' => route($route, $identifier),
            'id' => 'user_change_password'
        ]);

        $view->with('passwordForm', $passwordForm);

        return $view;
    }

    /**
     * overwritten function for handle modal form
     * @return $this
     */
    public function create()
    {
        $createForm = $this->form($this->formClass, [
            'method' => 'POST',
            'url' => route($this->routes['store']),
            'id' => 'module_form'
        ]);

        $createView = $this->views['create'];

        $mode = Request::get('mode', 'full');

        if ($mode == 'modal') {
            $createView = 'core::crud.module.create_form';
        }

        $view = view($createView);

        if ($mode == 'modal') {
            $formId = uniqid('form_');

            $view->with('modal_form', true);
            $createForm->setFormOption('id', $formId);

            $createForm->setFormOption('class', 'module_form related-modal-form row');

            $view->with('formId', $formId);
            $createForm->add('entityCreateMode', 'hidden', [
                'value' => 'modal'
            ]);
            $createForm->add('relationType', 'hidden', [
                'value' => Request::get('relationType')
            ]);
            $createForm->add('relatedField', 'hidden', [
                'value' => Request::get('relatedField')
            ]);
            $createForm->add('relatedEntityId', 'hidden', [
                'value' => Request::get('relatedEntityId')
            ]);

            $createForm->add('relatedEntity', 'hidden', [
                'value' => Request::get('relatedEntity')
            ]);
            $view->with('sectionButtons', []);

        }

        $view->with('form_request', $this->storeRequest);
        $view->with('show_fields', $this->showFields);

        return $view->with('form', $createForm);
    }

    /**
     * overwritten function to apply password
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function store()
    {
        if (config('bap.demo')) {
            flash(trans('core::core.you_cant_do_that_its_demo'))->error();
            return redirect()->back();
        }
        $request = \App::make($this->storeRequest ?? Request::class);

        $input = $request->all();

        foreach($input as $key => $i){
            if($i === ''){
                unset($input[$key]);
            }
        }

        $randomPassword = UserHelper::randomPassword(6);

        $input['password'] = bcrypt($randomPassword);

        // chefe cadastrando outros users
        if(!isset($dados['company_id']) && \Auth::user()->company_id != ''){
            $input['company_id'] = \Auth::user()->company_id;
        }

        $entity = \App::make($this->repository)->create($input);

        if (is_null($request->get('roles'))) {
            $entity->syncRoles([]);
        } else {
            $entity->syncRoles($request->get('roles'));
        }

        flash(trans($this->languageFile . '.created', ['password'=>$randomPassword]))->success();

        return redirect(route($this->routes['index']));
    }

    /**
     * Update Entity
     * overwritten function to apply syncRoles
     *
     * @param $identifier
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function update($identifier)
    {

        if (config('bap.demo')) {
            flash(trans('core::core.you_cant_do_that_its_demo'))->error();
            return redirect()->back();
        }

        $request = \App::make($this->updateRequest ?? Request::class);

        $repository = \App::make($this->repository);

        $entity = $repository->find($identifier);

        if (empty($entity)) {
            flash(trans($this->languageFile . '.entity_not_found'))->success();

            return redirect(route($this->routes['index']));
        }

        $input = $request->all();

        foreach($input as $key => $i){
            if($i === ''){
                unset($input[$key]);
            }
        }

        $entity = \App::make($this->repository)->update($input, $identifier);

        if (is_null($request->get('roles'))) {
            $entity->syncRoles([]);
        } else {
            $entity->syncRoles($request->get('roles'));
        }

        // UPLOAD PROFILE PICTURE
        $profilePicture = $request->file('profile_picture');

        if ($profilePicture != null) {
            $image = 'profile_' . $identifier . '_.' . $profilePicture->getClientOriginalExtension();

            $uploadSuccess = $profilePicture->move(UserHelper::PROFILE_PICTURE_UPLOAD, $image);

            if ($uploadSuccess) {

                // Resize uploaded image to 100x100
                $img = Image::make(base_path() . '/public/' . UserHelper::PROFILE_PICTURE_UPLOAD . $image)->resize(
                    100,
                    100
                );
                $img->save(base_path() . '/public/' . UserHelper::PROFILE_PICTURE_UPLOAD . $image);

                $entity = \App::make($this->repository)->update([
                    'profile_image_path' => $image
                ], $identifier);
            }
        }

        flash(trans($this->languageFile . '.updated'))->success();

        return redirect(route($this->routes['show'], $entity));
    }

    protected function setupCustomButtons()
    {
        $this->customShowButtons[] = array(
            'attr' => [
                'class' => 'btn bg-pink waves-effect',
                'data-toggle' => 'modal',
                'data-target' => '#defaultModal'
            ],
            'label' => trans('user::users.change_password')
        );

        if(\Auth::user()->hasRole('admin')){
            $this->customShowButtons[] = array(
                'attr' => [
                    'class' => 'btn bg-pink waves-effect',
                    'onclick' => 'BAP_Users.loginAsUser("' . $this->entity->id . '")',
                ],
                'label' => trans('user::users.ghost_login'),
            );
        }

        $this->customShowButtons[] = array(
            'href' => route('settings.users.activity', $this->entity->id),
            'attr' => [
                'class' => 'btn bg-pink waves-effect',
            ],
            'label' => trans('user::users.activity_log')
        );
    }
}
