<?php
namespace Modules\Companies\Datatables\Tabs;

use Modules\Collaborators\Entities\Collaborator;
use Modules\Companies\Entities\Company;
use Modules\Places\Entities\Place;
use Modules\Collaborators\Entities\Area;
use Modules\Platform\Core\Datatable\RelationDataTable;
use Modules\Platform\Core\Helper\DataTableHelper;
use Yajra\DataTables\EloquentDataTable;

/**
 * Class CampaignsContactsDatatable
 * @package Modules\Campaigns\Datatables\Tabs
 */
class CompanyCollaboratorsDatatable extends RelationDataTable
{
    const SHOW_URL_ROUTE = 'collaborators.collaborators.show';
    const EDIT_URL_ROUTE = 'collaborators.collaborators.edit';
    const REMOVE_URL_ROUTE = 'collaborators.collaborators.destroy';

    protected $unlinkRoute = 'companies.collaborators.unlink';

    /**
     * Build DataTable class.
     *
     * @param mixed $query Results from query() method.
     * @return \Yajra\DataTables\DataTableAbstract
     */
    public function dataTable($query)
    {
        $dataTable = new EloquentDataTable($query);

        $this->applyLinks($dataTable, self::SHOW_URL_ROUTE, self::EDIT_URL_ROUTE, self::REMOVE_URL_ROUTE, 'companies_');

        return $dataTable;
    }


    /**
     * Get query source of dataTable.
     *
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query(Collaborator $model)
    {
        $query = $model->newQuery()
        ->leftJoin('places', 'collaborators.place_id', '=', 'places.id')
        ->leftJoin('areas', 'collaborators.area_id', '=', 'areas.id')
        ->select([
            'collaborators.id',
            'collaborators.company_id',
            'collaborators.place_id',
            'collaborators.area_id',
            'collaborators.name',
            'collaborators.email',
            'collaborators.phone',
            'collaborators.mobile',
            'collaborators.city',
            'collaborators.state',
            'collaborators.person_tax_number',
            'places.identifier_name as place_name',
            'areas.name as area_name',
        ]);

        if(\Auth::user()->hasRole('gerente')){
            $query = $query->where('collaborators.place_id',\Auth::user()->place_id);
        }

        return $query;
    }


    /**
     * Optional method if you want to use html builder.
     *
     * @return \Yajra\DataTables\Html\Builder
     */
    public function html()
    {

        $currentColumn = 0;
        $columnFilters = [];

        if(\Auth::user()->hasRole('admin') || \Auth::user()->hasRole('chefe')){

            $places = [];
            $data_places = [];
            if (\Auth::user()->hasRole('admin')) {
                $places = Place::all()->toArray();
            }
            else {
                $places = Place::where('company_id', \Auth::user()->company_id)->get()->toArray();
            }
            foreach ($places as $place){
                $data_places[] = [
                    'value' => $place['id'],
                    'label' => $place['identifier_name'],
                ];
            }

            $columnFilters[] = [
                'column_number' => $currentColumn++,
                'filter_type' => 'select',
                'select_type' => 'select2',
                'select_type_options' => [
                    'theme' => "bootstrap",
                    'width' => '100%'
                ],
                'data' => $data_places                    
            ];
        }


        $areas = [];
        $data_areas = [];
        if (\Auth::user()->hasRole('admin')) {
            $areas = Area::all()->toArray();
        }
        else {
            $areas = Area::where('company_id', \Auth::user()->company_id)->get()->toArray();
        }
    
        foreach ($areas as $area){
            $data_areas[] = [
                'value' => $area['id'],
                'label' => $area['name'],
            ];
        }

        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'select',
            'select_type' => 'select2',
            'select_type_options' => [
                'theme' => "bootstrap",
                'width' => '100%'
            ],
            'data' => $data_areas                    
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
       
        return $this->builder()
            ->columns($this->getColumns())
            ->setTableId('CompanyCollaboratorsDatatable' . $this->tableSuffix)
            ->minifiedAjax(route($this->route, ['entityId' => $this->entityId]))

            ->setTableAttribute('class', 'table responsive table-hover')
            ->parameters([
                'dom' => "<'row'<'col-sm-12'tr>>\n\t\t\t<'row'<'col-sm-12 col-md-5'i><'col-sm-12 col-md-7 dataTables_pager'lp>>",

                'stateSave' => true,
                'columnFilters' => $columnFilters,
                'buttons' => DataTableHelper::buttons(),
                'regexp' => true

            ]);
    }

    /**
     * @return array
     */
    protected function getColumns()
    {

        $unlink = [
            'unlink' => [
                'data' => 'unlink',
                'title' => '',
                'data_type' => 'unlink',
                'orderable' => false,
                'searchable' => false,

            ]
        ];

        $check_selection = [
            'check_select' => [
                'data' => 'check_select',
                'title' => '',
                'data_type' => 'check_select',
                'orderable' => false,
                'searchable' => false,
            ]
        ];

        $data = [];

        if(\Auth::user()->hasRole('admin') || \Auth::user()->hasRole('chefe')){
            $data += [
                'place_name' => [
                    'name' => 'place_id',
                    'data' => 'place_name',
                    'title' => trans('suppliers::suppliers.table.place_name'),
                    'data_type' => 'text'
                ],
            ];
        }

        $data += [

            'area_name' => [
                'name' => 'area_id',
                'data' => 'area_name',
                'title' => trans('collaborators::collaborators.table.area_name'),
                'data_type' => 'text'
            ],
            'name' => [
                'data' => 'name',
                'title' => trans('collaborators::collaborators.table.name'),
                'data_type' => 'text'
            ],
            'email' => [
                'data' => 'email',
                'title' => trans('collaborators::collaborators.table.email'),
                'data_type' => 'text'
            ],
            'actions' => ['title' => trans('core::core.table.actions'), 'data_type' => 'actions', 'orderable' => false],
        ];

        return $data;
    }
}
