<?php

namespace Modules\Asset\Service;

use DateTime;
use Modules\Assets\Entities\Asset;
use Modules\Places\Entities\Place;
use Modules\Assets\Entities\Status;
use Modules\Assets\Entities\Category;
use Modules\Companies\Entities\Company;
use Spatie\Activitylog\Models\Activity;
use Modules\Assets\Entities\AssetStatusLog;

/**
 * Class AssetService
 * @package Modules\Asset\Service
 */
class AssetService {

    /**
     * Count of assets according to access rules
     *
     * @return mixed
     */
    public function count() {

        if(\Auth::user()->hasRole('admin')){

            return Asset::count();
        }

        elseif(\Auth::user()->hasRole('chefe')){

            return Asset::where('company_id',\Auth::user()->company_id)->count();
        }

        elseif(\Auth::user()->hasRole('gerente')){

            return Asset::where('place_id',\Auth::user()->place_id)->count();
        }

    }

    /**
     * Count of assets according to some place
     *
     * @return integer
     */
    public function countByPlace($placeId) {

        return Asset::where('place_id',$placeId)->count();

    }

    /**
     * Count of assets according to some company
     *
     * @return integer
     */
    public function countByCompany($companyId) {

        return Asset::where('company_id',$companyId)->count();

    }

    /**
     * Count of assets according to some company
     *
     * @return integer
     */
    public function countByCategoryAndCompany($categoryId,$companyId) {

        return Asset::where([
            ['category_id',$categoryId],
            ['company_id', $companyId]
        ])->count();

    }
    
    /**
     * Count of assets according to conservation state and place
     *
     * @return integer
     */
    public function countByConservationStateAndPlace($conservationState,$placeId) {

        return Asset::where([
            ['conservation_state',$conservationState],
            ['place_id', $placeId]
        ])->count();

    }

    /**
     * Count of assets according to some company
     *
     * @return integer
     */
    public function countByCategoryAndPlace($categoryId,$placeId) {

        return Asset::where([
            ['category_id',$categoryId],
            ['place_id', $placeId]
        ])->count();

    }

    /**
     * Count of assets according to some company
     *
     * @return integer
     */
    public function countByCategory($categoryId) {

        return Asset::where('category_id',$categoryId)->count();

    }

    /**
     * General count of assets
     *
     * @return integer
     */
    public function generalCount() {

        return Asset::all()->count();

    }

    /**
     * Returns data of assets by category
     *
     * @return array
     */
    public function getByCategory($companyId = null, $placeId = null) {

        $result = [];

        $colors = [
            'red','blue','violet','green','brown','purple','yellow','black','gray','orange','pink'
        ];

        $colorCount = 0;
        $categories = Category::all();

        if($placeId != null){
            $assetsCount = $this->countByPlace($placeId);    
        }
        elseif($companyId != null){
            $assetsCount = $this->countByCompany($companyId);
        }
        else{
            $assetsCount = $this->count();
        }

        foreach($categories as $category){

            $countAssetCategory = 0;

            if($placeId != null){

                $countAssetCategory = Asset::where([
                    ['place_id', $placeId ],
                    ['category_id', $category['id']]
                ])->count();
            }
            elseif($companyId != null){

                $countAssetCategory = Asset::where([
                    ['company_id', $companyId ],
                    ['category_id', $category['id']]
                ])->count();
            }
            else{

                if(\Auth::user()->hasRole('admin')){

                    $countAssetCategory = Asset::where('category_id', $category['id'])->count();
                }
                elseif(\Auth::user()->hasRole('chefe')) {
    
                    $countAssetCategory = Asset::where([
                        ['company_id', \Auth::user()->company_id ],
                        ['category_id', $category['id']]
                    ])->count();
                }
                elseif(\Auth::user()->hasRole('gerente')) {
    
                    $countAssetCategory = Asset::where([
                        ['place_id', \Auth::user()->place_id ],
                        ['category_id', $category['id']]
                    ])->count();
                }
            }

            $result[] = [
                'name' => $category['name'],
                'percentage' => ($assetsCount > 0 && $countAssetCategory > 0) ? number_format((($countAssetCategory * 100) / $assetsCount), 2, '.', '') : 0,
                'color' => $colors[$colorCount++],
                'count' => $countAssetCategory
            ];
            ($colorCount == count($colors) -1) ? $colorCount = 0 : $colorCount = $colorCount;
        }

        return $result;
    }

    /**
     * Returns data of assets by status
     *
     * @return array
     */
    public function getByStatus($companyId = null, $placeId = null) {

        $status = Status::all();

        $data = [];
        $dataLabels = [];
        $dataStatus = [];

        $currentMonth = date('m');
        $currentYear = date('Y');

        for($x=0;$x<12;$x++){

            if(strlen($currentMonth) == 1){
                $currentMonth = '0'.$currentMonth;
            }
            $dataLabels[] = $currentMonth . '/'. substr($currentYear, -2);

            foreach($status as $st){

                if($x == 0){

                    if(\Auth::user()->hasRole('admin')){
                        $dataStatus[] = $st['name']. ' - '.Company::find($st['company_id'])['name'];
                    }
                    else{
                        $dataStatus[] = $st['name'];
                    }

                }

                $dataCurrentMonth = AssetStatusLog::
                    whereRaw('MONTH(created_at) = ?',[$currentMonth])
                    ->where('status_id', '=', $st['id']);

                if($companyId == null && $placeId == null){
                    
                    if(\Auth::user()->hasRole('chefe')){
                        $dataCurrentMonth = $dataCurrentMonth->where('company_id', '=', \Auth::user()->company_id);
                    }
                    if(\Auth::user()->hasRole('gerente')){
                        $dataCurrentMonth = $dataCurrentMonth->where('place_id', '=', \Auth::user()->place_id);
                    }
                }
                elseif($placeId != null){ 

                    $dataCurrentMonth = $dataCurrentMonth->where('place_id', '=', $placeId);
                }
                elseif($companyId != null){

                    $dataCurrentMonth = $dataCurrentMonth->where('company_id', '=', $companyId);
                }

                $dataCurrentMonth = $dataCurrentMonth->orderBy('created_at','desc');

                if($companyId == null && $placeId == null){
                    if(\Auth::user()->hasRole('admin')){
                        $dataCurrentMonth = $dataCurrentMonth->where('company_id', '!=', '')->take(Company::all()->count());
                    }
                    else{
                        $dataCurrentMonth = $dataCurrentMonth->take(1);
                    }
                }
                else{
                    $dataCurrentMonth = $dataCurrentMonth->take(1);
                }
                $dataCurrentMonth = $dataCurrentMonth->get();
 
                if(count($dataCurrentMonth) == 0){
                    $data[] = [
                        'count' => 0,
                    ];
                }
                else{
                    $count = 0;
                    foreach($dataCurrentMonth as $d){
                        $count +=  $d->quantity;
                    }

                    $data[] = [
                        'count' => $count,
                    ];
                }

            }

            if($currentMonth == 1) {
                $currentMonth = 12;
                $currentYear--;
            }
            else{
                $currentMonth--;
            }
        }

        $data['labels'] = $dataLabels;
        $data['status'] = $dataStatus;
        $data['status_quantity'] = count($status);
        $data['bars_quantity'] = 1;
 
        return $data;

    }

    /**
     * Returns data of assets by status
     *
     * @return array
     */
    public function getByCompanies() {

        $result = [];

        $colors = [
            'blue','red','violet','green','brown','purple','yellow','black','gray','orange','pink'
        ];

        $colorCount = 0;

        $assetsCount = $this->count();

        $companies = Company::all();

        foreach($companies as $company){

            $countCompanyAssets = Asset::where('company_id', $company['id'])->count();
 
            $result[] = [
                'name' => $company['name'],
                'count' => $countCompanyAssets,
                'color' => $colors[$colorCount++],
                'percentage' => ($assetsCount > 0 && $countCompanyAssets > 0) ? number_format((($countCompanyAssets * 100) / $assetsCount), 2, '.', '') : 0,
            ];
            ($colorCount == count($colors) -1) ? $colorCount = 0 : $colorCount = $colorCount;
        }

        return $result;
    }

    /**
     * Returns data of assets by status and place
     *
     * @return array
     */
    public function getByStatusAndPlace($placeId) {

        $result = [];

        $colors = [
            'red','blue','violet','green','brown','purple','yellow','black','gray','orange','pink'
        ];

        $colorCount = 0;

        $status = Status::where('company_id', Place::find($placeId)->company_id)->get();

        foreach($status as $stat){

            $countAssetStatus = Asset::where([
                ['place_id', $placeId],
                ['status_id', $stat['id']]
            ])->count();

            $result[] = [
                'name' => $stat['name'],
                'count' => $countAssetStatus,
                'color' => "#00c5dc",//$colors[$colorCount++],
            ];
            ($colorCount == count($colors) -1) ? $colorCount = 0 : $colorCount = $colorCount;
        }

        return $result;
    }

    /**
     * Get last updates from assets.
     *
     * @return array
     */
    public function lastUpdates($filter = null, $identifier = null) { 

        $activity = new Activity();

        $activity = $activity->newQuery()->with('causer')
        ->leftJoin('users', 'activity_log.causer_id', '=', 'users.id')
        ->leftJoin('assets', 'activity_log.subject_id', '=', 'assets.id')
        ->leftJoin('places', 'assets.place_id', '=', 'places.id')
        ->leftJoin('areas', 'assets.area_id', '=', 'areas.id')
        ->leftJoin('status', 'assets.status_id', '=', 'status.id')
        ->select([
            'activity_log.id',
            'activity_log.log_name',
            'activity_log.description',
            'activity_log.subject_id',
            'activity_log.subject_type',
            'activity_log.causer_id',
            'activity_log.causer_type',
            'activity_log.properties',
            'activity_log.created_at',
            'activity_log.updated_at',
            'users.name as user_name',
            'places.identifier_name as asset_place',
            'areas.name as asset_area',
            'status.name as asset_status',
        ])->where('subject_type','Modules\Assets\Entities\Asset');

        if($filter == null){
            if(\Auth::user()->hasRole('chefe')){

                $activity = $activity->where('users.company_id',\Auth::user()->company_id);
            }
            elseif(\Auth::user()->hasRole('gerente')){

                $activity = $activity->where('users.place_id',\Auth::user()->place_id);
            }
        }
        else{
            if($filter == 'company'){

                $activity = $activity->where('users.company_id',$identifier);
            }
            elseif($filter == 'place'){

                $activity = $activity->where('users.place_id',$identifier);
            }
        }

        $activity = $activity->orderBy('updated_at', 'desc')
        ->take(15)
        ->get();

        return $activity;
    }

    /**
     * Return the age in month from assets
     */
    public function getAge(Asset $asset){

        if($asset->purchase_date != ''){

            $diff = (new DateTime())->diff(new DateTime($asset->purchase_date));

            return (($diff->format('%y') * 12) + $diff->format('%m'));
        }
        else {

            return '';
        }
    }

    /**
     * Return the current value from assets according to some category
     */
    public function getCurrentValueByCategory($category,$companyId = null, $placeId = null){

        if($placeId != null){
            $assets = Asset::where([
                ['category_id', $category['id']],
                ['place_id', $placeId]
            ])->get();
        }
        elseif($companyId != null){
            $assets = Asset::where([
                ['category_id', $category['id']],
                ['company_id', $companyId]
            ])->get();
        }
        else{
            $assets = Asset::where('category_id', $category['id'])->get();
        }

        $current_value = 0;

        foreach($assets as $asset){
            $current_value += self::getCurrentValue($asset);
        }

        return number_format($current_value, 2);

    }

    /**
     * Return the average value from assets according to some category
     */
    public function getAverageCostByCategory($category,$companyId = null, $placeId = null){

        $purchase_value = self::getPurchaseValueByCategory($category,$companyId,$placeId);

        $purchase_value = str_replace(',','',$purchase_value);

        if($category['assets_quantity'] > 0){
            return number_format( $purchase_value / $category['assets_quantity'], 2);
        }
        else{
            return 0;
        }
    }

    /**
     * Return the current value from assets according to some category
     */
    public function getPurchaseValueByCategory($category, $companyId = null, $placeId = null){

        if($placeId != null){
            $assets = Asset::where([
                ['category_id', $category['id']],
                ['place_id', $placeId]
            ])->get();
        }
        elseif($companyId != null){
            $assets = Asset::where([
                ['category_id', $category['id']],
                ['company_id', $companyId]
            ])->get();
        }
        else{
            $assets = Asset::where('category_id', $category['id'])->get();
        }

        $purchase_value = 0;

        foreach($assets as $asset){
            if($asset['purchase_value'] != ''){
                $asset_purchase_value = str_replace(',','.',$asset['purchase_value']);
                $purchase_value += $asset_purchase_value;
            }
        }

        return number_format($purchase_value, 2);

    }

    /**
     * Return the current value from assets
     */
    public function getCurrentValue(Asset $asset){

        if ( $asset->purchase_value != '' && $asset->useful_life != '0') {

            $asset->purchase_value = str_replace(',','.',$asset->purchase_value);

            $diff = (new DateTime())->diff(new DateTime($asset->purchase_date));

            $age = $asset->useful_life - ($diff->y + $diff->m / 12 + $diff->d / 365);

            if($asset->conservation_state != ''){
                if($asset->conservation_state == 'perfeito'){
                    $current_value = round($asset->purchase_value  * ($age / $asset->useful_life) , 2);
                    return $current_value;
                }
                elseif($asset->conservation_state == 'bom'){
                    $current_value = round(($asset->purchase_value  * ($age / $asset->useful_life) ) * 0.8, 2);
                    return $current_value;
                }
                elseif($asset->conservation_state == 'regular'){
                    $current_value = round(($asset->purchase_value  * ($age / $asset->useful_life) ) * 0.6, 2);
                    return $current_value;
                }
                elseif($asset->conservation_state == 'ruim'){
                    $current_value = round(($asset->purchase_value  * ($age / $asset->useful_life) ) * 0.4, 2);
                    return $current_value;
                }
                elseif($asset->conservation_state == 'muito ruim'){
                    $current_value = round(($asset->purchase_value  * ($age / $asset->useful_life) ) * 0.2, 2);
                    return $current_value;
                }
            }
            else{
                $current_value = round($asset->purchase_value / 10 * $age, 2);
                return $current_value;
            }

        }
        else {

            return '0';
        }

    }

    /**
     * Returns data of assets by conservation state
     *
     * @return array
     */
    public function getByConservationStateAndCategories($companyId = null, $placeId = null) {

        $result = [];

        $colors = [
            'perfeito' => '#228B22',
            'bom' => '#7CFC00',
            'regular' => '#FFFF00'
            ,'ruim' => '#FF8C00',
            'muito ruim' => '#FF4500',
        ];

        $categories = Category::all();

        $conservationStates = ['perfeito', 'bom', 'regular', 'ruim', 'muito ruim'];

        foreach($categories as $category){

            if($placeId != null){

                $assetsCount = $this->countByCategoryAndPlace($category['id'],$placeId);
            }
            elseif($companyId != null){

                $assetsCount = $this->countByCategoryAndCompany($category['id'],$companyId);
            }
            else{

                $assetsCount = $this->countByCategory($category['id']);
            }

            $result[$category['name']]['count'] = 0;

            foreach($conservationStates as $conservationState){

                $countAssetConservationState = 0;

                if($placeId != null){

                    $countAssetConservationState = Asset::where([
                        ['place_id', $placeId ],
                        ['conservation_state', $conservationState],
                        ['category_id', $category['id']]
                    ])->count();
                }
                elseif($companyId != null){

                    $countAssetConservationState = Asset::where([
                        ['company_id', $companyId ],
                        ['conservation_state', $conservationState],
                        ['category_id', $category['id']]
                    ])->count();
                }
                else{

                    if(\Auth::user()->hasRole('admin')){

                        $countAssetConservationState = Asset::where([
                            ['conservation_state', $conservationState],
                            ['category_id', $category['id']]
                        ])->count();
                    }
                    elseif(\Auth::user()->hasRole('chefe')) {
    
                        $countAssetConservationState = Asset::where([
                            ['company_id', \Auth::user()->company_id ],
                            ['conservation_state', $conservationState],
                            ['category_id', $category['id']]
                        ])->count();
                    }
                    elseif(\Auth::user()->hasRole('gerente')) {
    
                        $countAssetConservationState = Asset::where([
                            ['place_id', \Auth::user()->place_id ],
                            ['conservation_state', $conservationState],
                            ['category_id', $category['id']]
                        ])->count();
                    }
                }

                $result[$category['name']][] = [
                    'name' => $conservationState,
                    'percentage' => ($assetsCount > 0 && $countAssetConservationState > 0) ? number_format((($countAssetConservationState * 100) / $assetsCount), 2, '.', '') : 0,
                    'color' => $colors[$conservationState],
                ];
                $result[$category['name']]['count'] += $countAssetConservationState;
            }
            $result[$category['name']]['category'] = $category['id'];

        }

        return $result;
    }

    /**
     * Returns data of assets by conservation state
     *
     * @return array
     */
    public function getByConservationState($companyId = null, $placeId = null) {

        $result = [];

        $colors = [
            'perfeito' => '#228B22',
            'bom' => '#7CFC00',
            'regular' => '#FFFF00'
            ,'ruim' => '#FF8C00',
            'muito ruim' => '#FF4500',
        ];

        if($placeId != null){

            $assetsCount = $this->countByPlace($placeId);
        }
        elseif($companyId != null){

            $assetsCount = $this->countByCompany($companyId);
        }
        else{

            $assetsCount = $this->count();
        }

        $conservationStates = ['perfeito', 'bom', 'regular', 'ruim', 'muito ruim'];

        foreach($conservationStates as $conservationState){

            $countAssetConservationState = 0;

            if($placeId != null){

                $countAssetConservationState = Asset::where([
                    ['conservation_state', $conservationState],
                    ['place_id',$placeId]
                ])->count();
            }
            elseif($companyId != null){

                $countAssetConservationState = Asset::where([
                    ['conservation_state', $conservationState],
                    ['company_id',$companyId]
                ])->count();
            }
            else{

                if(\Auth::user()->hasRole('admin')){

                    $countAssetConservationState = Asset::where('conservation_state', $conservationState)->count();
                }
                elseif(\Auth::user()->hasRole('chefe')) {
    
                    $countAssetConservationState = Asset::where([
                        ['company_id', \Auth::user()->company_id ],
                        ['conservation_state', $conservationState]
                    ])->count();
                }
                elseif(\Auth::user()->hasRole('gerente')) {
    
                    $countAssetConservationState = Asset::where([
                        ['place_id', \Auth::user()->place_id ],
                        ['conservation_state', $conservationState]
                    ])->count();
                }
            }

            $result[] = [
                'name' => $conservationState,
                'percentage' => ($assetsCount > 0 && $countAssetConservationState > 0) ? number_format((($countAssetConservationState * 100) / $assetsCount), 2, '.', '') : 0,
                'color' => $colors[$conservationState],
                'count' => $countAssetConservationState
            ];
        }

        return $result;
    }

    /**
     * Returns data of assets by category
     *
     * @return array
     */
    public function getByCategoryCosts($companyId = null, $placeId = null) {

        $result = [];

        $colors = [
            'blue','violet','green','brown','purple','yellow','black','gray','orange','pink'
        ];

        $colorCount = 0;

        $categories = Category::all();

        $totalAssetsValue = self::getTotalCost($companyId,$placeId);
        $totalAssetsValue = str_replace(',','',$totalAssetsValue);

        foreach($categories as $category){

            $countAssetCategory = 0;

            $categoryValue = self::getCurrentValueByCategory($category,$companyId,$placeId);

            $result[] = [
                'name' => $category['name'],
                'percentage' => ($categoryValue > 0 && $totalAssetsValue > 0) ? number_format(((str_replace(',','',$categoryValue) * 100) / $totalAssetsValue), 2, '.', '') : 0,
                'color' => $colors[$colorCount++],
                'count' => $categoryValue
            ];
            ($colorCount == count($colors) -1) ? $colorCount = 0 : $colorCount = $colorCount;
        }

        return $result;
    }

    /**
     * Returns the total cost
     */
    public function getTotalCost($companyId = null, $placeId = null){

        $categories = Category::all();

        $totalCost = 0;

        foreach($categories as $category){

            if($companyId != null){
                $categoryValue = self::getCurrentValueByCategory($category,$companyId);
            }
            elseif($placeId != null){
                $categoryValue = self::getCurrentValueByCategory($category,null,$placeId);
            }
            else{
                $categoryValue = self::getCurrentValueByCategory($category);
            }
    
            $totalCost += str_replace(',','',$categoryValue);
        }

        return number_format($totalCost, 2);
    }

    /**
     * Returns data of costs from assets
     *
     * @return array
     */
    public function getCosts($companyId = null, $placeId = null) {

        $categories = Category::all()->toArray();

        $data = [];
        $dataLabels = [];
        $dataCosts = [];

        foreach($categories as $category){

            $dataLabels[] = $category['name'];

            if($companyId == null && $placeId == null){

                if(\Auth::user()->hasRole('chefe')){
                    $dataCost = self::getPurchaseValueByCategory($category,\Auth::user()->company_id);
                }
                elseif(\Auth::user()->hasRole('gerente')){
                    $dataCost = self::getPurchaseValueByCategory($category,null,\Auth::user()->place_id);
                }
                else{
                    $dataCost = self::getPurchaseValueByCategory($category);
                }
            }

            elseif($placeId != null){ 

                $dataCost = self::getPurchaseValueByCategory($category,null,$placeId);
            }
            elseif($companyId != null){

                $dataCost = self::getPurchaseValueByCategory($category,$companyId);
            }

            $data[] = [
                'value' => $dataCost,
            ];
        }

        $data['labels'] = $dataLabels;

        return $data;

    }

}
