<?php

namespace Modules\Assets\Http\Controllers\Settings;

use M1\Env\Helper\StringHelper;
use Modules\Assets\Entities\Status;
use Modules\Platform\Core\Helper\ValidationHelper;
use Modules\Assets\Datatables\Settings\StatusDatatable;
use Modules\Platform\Core\Http\Forms\NameDictionaryForm;
use Modules\Platform\Core\Repositories\GenericRepository;
use Modules\Platform\Core\Http\Requests\NameDictionaryRequest;
use Modules\Platform\Core\Http\Controllers\ModuleCrudController;
use Modules\Platform\Notifications\Entities\NotificationPlaceholder;

class StatusController extends ModuleCrudController
{

    protected $datatable = StatusDatatable::class;
    protected $formClass = NameDictionaryForm::class;
    protected $storeRequest = NameDictionaryRequest::class;
    protected $updateRequest = NameDictionaryRequest::class;
    protected $entityClass = Status::class;

    protected $moduleName = 'assets';

    protected $settingName = 'status';

    protected $settingsBackRoute = 'assets.assets.index';

    protected $showFields = [
        'details' => [
            'name' => ['type' => 'text'],
            'notes' => ['type' => 'text'],
        ]
    ];

    protected $languageFile = 'assets::assets.status';

    protected $routes = [
        'index' => 'assets.status.index',
        'create' => 'assets.status.create',
        'show' => 'assets.status.show',
        'edit' => 'assets.status.edit',
        'store' => 'assets.status.store',
        'destroy' => 'assets.status.destroy',
        'update' => 'assets.status.update'
    ];

    public function __construct()
    {
        parent::__construct();
    }


    /**
     * Overwriten function for attach status on company
     *
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function store() {

        $request = \App::make($this->storeRequest ?? Request::class);

        $mode = $request->get('entityCreateMode', self::FORM_MODE_FULL);

        if ($this->permissions['create'] != '' && !\Auth::user()->hasPermissionTo($this->permissions['create'])) {
            if ($mode == self::FORM_MODE_SIMPLE) {
                return response()->json([
                    'type' => 'error',
                    'message' => trans('core::core.entity.you_dont_have_access'),
                    'action' => 'show_message'
                ]);
            }
            flash(trans('core::core.you_dont_have_access'))->error();
            return redirect()->route($this->routes['index']);
        }

        $repository = $this->getRepository();

        $storeValues = $this->form($this->formClass)->getFieldValues(true);

        if ($mode == self::FORM_MODE_SIMPLE) {

            //Bind related element
            $relatedEntityId = $request->get('relatedEntityId');
            $relationType = $request->get('relationType', null);
            $relatedField = $request->get('relatedField');
            $relatedEntity = $request->get('relatedEntity');

            if ($relationType != null) { // Relation type is not null

                $relationEntityRepos = \App::make(GenericRepository::class);
                $relationEntityRepos->setupModel($relatedEntity);

                $relationEntity = $relationEntityRepos->findWithoutFail($relatedEntityId);

                if ($relationType == 'oneToMany') {
                    $storeValues[$relatedField] = $relationEntity->id;
                }
            }
        }

        $input = [];
        foreach($storeValues as $key => $value){
            if($value === '' || $key == 'company_id'){
                unset($storeValues[$key]);
            }
        }


        $entity = $repository->createEntity($storeValues, \App::make($this->entityClass));


        $entity = $this->setupAssignedTo($entity, $request, true);
        $entity->save();

        /**
         * Attach status on company
         */
        $entity->companies()->attach($request->input('company_id'));

        if (config('bap.record_assigned_notification_enabled')) {

            if ($entity instanceof Ownable) {
                if ($entity->getOwner() != null && $entity->getOwner() instanceof User) {
                    if ($entity->getOwner()->id != \Auth::user()->id) { // Dont send notification for myself
                        try {
                            $commentOn = $entity->name;
                            $commentOn = ' - ' . $commentOn;
                        } catch (\Exception $exception) {
                            $commentOn = '';
                        }

                        $placeholder = new NotificationPlaceholder();

                        $placeholder->setRecipient($entity->getOwner());
                        $placeholder->setAuthorUser(\Auth::user());
                        $placeholder->setAuthor(\Auth::user()->name);
                        $placeholder->setColor('bg-green');
                        $placeholder->setIcon('assignment');
                        $placeholder->setContent(trans('notifications::notifications.new_record', ['user' => \Auth::user()->name]) . $commentOn);

                        $placeholder->setUrl(route($this->routes['show'], $entity->id));

                        $entity->getOwner()->notify(new GenericNotification($placeholder));
                    }
                }
            }
        }

        if ($mode == self::FORM_MODE_SIMPLE) {

            //Bind related element
            $relatedEntityId = $request->get('relatedEntityId');
            $relationType = $request->get('relationType', null);
            $relatedField = $request->get('relatedField');
            $relatedEntity = $request->get('relatedEntity');

            if ($relationType != null) { // Relation type is not null

                $relationEntityRepos = \App::make(GenericRepository::class);
                $relationEntityRepos->setupModel($relatedEntity);


                $relationEntity = $relationEntityRepos->findWithoutFail($relatedEntityId);


                if ($relationType == 'manyToMany') {
                    $entity->{$relatedField}()->attach($relationEntity->id);
                }
            }

            return response()->json([
                'type' => 'success',
                'message' => trans('core::core.entity.created'),
                'action' => 'refresh_datatable'
            ]);
        }

        if(! isset($this->settingName) ){
            $msg = $this->moduleName. '::' .$this->moduleName.'.created';
        }
        else{
            $msg = $this->moduleName. '::' .$this->moduleName.'.'.$this->settingName.'.created';
        }

        flash(trans($msg))->success();

        return redirect(route($this->routes['index']));
    }


    /**
     * Overwriten function for attach status on company
     *
     * @param $identifier
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function destroy($identifier) {

        if ($this->permissions['destroy'] != '' && !\Auth::user()->hasPermissionTo($this->permissions['destroy'])) {
            flash(trans('core::core.you_dont_have_access'))->error();
            return redirect()->route($this->routes['index']);
        }

        $repository = $this->getRepository();

        $entity = $repository->find($identifier);

        $this->entity = $entity;

        if (empty($entity)) {
            flash(trans('core::core.entity.entity_not_found'))->error();
            return redirect(route($this->routes['index']));
        }

        if ($this->blockEntityOwnableAccess()) {
            flash(trans('core::core.you_dont_have_access'))->error();
            return redirect()->route($this->routes['index']);
        }

        if (config('bap.validate_fk_on_soft_delete')) {
            $validator = ValidationHelper::validateForeignKeys($entity);

            if (count($validator) > 0) {
                flash(trans('core::core.cant_delete_check_fk_keys', ['fk_keys' => StringHelper::validationArrayToString($validator)]))->error();
                return redirect(route($this->routes['index']));
            }
        }

        try {

            /**
             * Detach companies
             */
            $entity->companies()->detach();

            $repository->delete($entity->id);
        } catch(\Exception $exception){
            flash("Impossível deletar, possui dependências")->error();

            return back();
        }

        if(! isset($this->settingName) ){
            $msg = $this->moduleName. '::' .$this->moduleName.'.deleted';
        }
        else{
            $msg = $this->moduleName. '::' .$this->moduleName.'.'.$this->settingName.'.deleted';
        }
        
        flash(trans($msg))->success();

        return redirect(route($this->routes['index']));
    }

}
