<?php

namespace Modules\Assets\Http\Controllers;

use DateTime;
use Illuminate\Http\Request;
use Modules\Assets\Entities\Asset;
use Intervention\Image\Facades\Image;
use Modules\Assets\Entities\Category;
use Modules\Asset\Service\AssetService;
use Modules\Assets\Http\Forms\AssetForm;
use Modules\Platform\Core\Helper\UserHelper;
use Modules\Assets\Datatables\AssetDatatable;
use Modules\Platform\User\Service\UserService;
use Modules\Assets\Http\Requests\AssetsRequest;
use Modules\Core\Notifications\GenericNotification;
use Modules\Platform\Core\Datatable\ActivityLogDataTable;
use Modules\Platform\Core\Repositories\GenericRepository;
use Modules\Platform\Core\Http\Controllers\ModuleCrudController;
use Modules\Platform\Notifications\Entities\NotificationPlaceholder;
use Modules\Assets\Datatables\Tabs\AssetMaintenancesDatatable;

  
class AssetsController extends ModuleCrudController
{

    protected $datatable = AssetDatatable::class;
    protected $formClass = AssetForm::class;
    protected $storeRequest = AssetsRequest::class;
    protected $updateRequest = AssetsRequest::class;
    protected $entityClass = Asset::class;

    protected $moduleName = 'assets';

    protected $jsFiles = [
        'BAP_Assets.js',
    ];

    protected $permissions = [
        'browse' => 'assets.browse',
        'create' => 'assets.create',
        'update' => 'assets.update',
        'destroy' => 'assets.destroy'
    ];

    protected $moduleSettingsLinks = [
        [
            'route' => 'assets.status.index', 
            'label' => 'settings.status'
        ],
        [
            'route' => 'assets.categories.index', 
            'label' => 'settings.categories'
        ],
    ];

    protected $relationTabs = [
        'maintenances'         => [
            'icon'        => 'compare_arrows',
            'permissions' => [
                'browse' => 'maintenances.browse',
                'update' => 'maintenances.update',
                'create' => 'maintenances.create',
            ],
            'datatable'   => [
                'datatable' => AssetMaintenancesDatatable::class,
            ],
            'route'       => [
                'linked'        => 'assets.maintenances.linked',
                'create'        => 'maintenances.maintenances.create',
                'select'        => 'assets.maintenances.selection',
                'bind_selected' => 'assets.maintenances.link',
            ],
            'create'      => [
                'allow'            => true,
                'modal_title'      => 'maintenances::maintenances.create_new',
                'post_create_bind' => [
                    'relationType' => 'oneToMany',
                    'relatedField' => 'asset_id',
                ],
            ],
            'select'      => [
                'allow'       => false,
                'modal_title' => 'maintenances::maintenances.module',
            ],
        ],
    ];

    protected $settingsPermission = 'collaborators.settings';

    protected $showFields = [

        'information' => [

            'company_id' => [
                'type' => 'hidden',
                'col-class' => 'col-12'
            ],

            'place_id' => [
                'type' => 'manyToOne',
                'relation' => 'Place',
                'column' => 'identifier_name',
                'col-class' => 'col-12'
            ],
            'area_id' => [
                'type' => 'manyToOne',
                'relation' => 'Area',
                'column' => 'name',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],

            'asset_number' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],

            'serial_number_1' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],

            'serial_number_2' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],

            'status_id' => [
                'type' => 'manyToOne',
                'relation' => 'Status',
                'column' => 'name',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],

            'category_id' => [
                'type' => 'manyToOne',
                'relation' => 'Category',
                'column' => 'name',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],

            'item' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],

            'model' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],

            'responsible' => [
                'type' => 'manyToOne',
                'relation' => 'collaborator',
                'column' => 'name',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],

            'supplier_id' => [
                'type' => 'manyToOne',
                'relation' => 'supplier',
                'column' => 'fantasy_name',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],

            'purchase_date' => [
                'type' => 'date',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],

            'purchase_value' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],

            'conservation_state' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],

            'useful_life' => [
                'type' => 'text',
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],
            'maintenance_total'      => [
                'type'      => 'decimal',
                'hide_in_form' => true,
                'show_in_form' => false,
                'col-class' => 'col-xl-6 col-lg-6 col-md-6 col-12'
            ],

            'image_path' => [
                'type' => 'image',
                'col-class' => 'col-md-12',
                'col-class' => 'col-12'
            ],
            
        ],

        'notes' => [

            'notes' => [
                'type' => 'text',
                'col-class' => 'col-12'
            ],

        ],

    ];

    protected $languageFile = 'assets::assets';

    protected $routes = [
        'index' => 'assets.assets.index',
        'create' => 'assets.assets.create',
        'show' => 'assets.assets.show',
        'edit' => 'assets.assets.edit',
        'store' => 'assets.assets.store',
        'destroy' => 'assets.assets.destroy',
        'update' => 'assets.assets.update'
    ];

    public function __construct() {
        parent::__construct();
    }

    /**
     * Overwritten function for add image and handle special fields from categories
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function store() {

        $request = \App::make($this->storeRequest ?? Request::class);

        $mode = $request->get('entityCreateMode', self::FORM_MODE_FULL);

        $input = $request->all();

        if(isset($input['purchase_value']) && $input['purchase_value'] != ''){
            $input['purchase_value'] = str_replace('.','',$input['purchase_value']);
        }

        $input['category_special_fields'] = '';

        if($input['category_id'] != ''){
            $category = Category::find($input['category_id']);
            $specialFields = explode(';;',$category->special_fields);
            if(is_array($specialFields) && count($specialFields) > 0){
                foreach($specialFields as $specialField){
                    if(isset($input['special_field_'.str_replace(' ','++',$specialField)])){
                        $input['category_special_fields'] .= $input['special_field_'.str_replace(' ','++',$specialField)].';;';
                    }
                }
            }
        }

        foreach($input as $key => $value){
            if($value === ''){
                unset($input[$key]);
            }
        }

        if(isset($input['useful_life']) && $input['useful_life'] != ''){
            $input['useful_life'] = str_replace(',','.',$input['useful_life']);
        }

        $entity = \App::make(Asset::class)->create($input);

        $assetImage = $request->file('image_path');

        if ($assetImage != null) {
            $image = 'asset_' . $entity->id . '_.' . $assetImage->getClientOriginalExtension();

            $uploadSuccess = $assetImage->move(UserHelper::ASSET_IMAGE_UPLOAD, $image);

            if ($uploadSuccess) {

                $img = Image::make(base_path() . '/public/' . UserHelper::ASSET_IMAGE_UPLOAD . $image);

                $img->save(base_path() . '/public/' . UserHelper::ASSET_IMAGE_UPLOAD . $image);

                $entity->update([
                    'image_path' => $image
                ]);
            }
        }

        if ($mode == self::FORM_MODE_SIMPLE) {

            //Bind related element
            $relatedEntityId = $request->get('relatedEntityId');
            $relationType = $request->get('relationType', null);
            $relatedField = $request->get('relatedField');
            $relatedEntity = $request->get('relatedEntity');

            if ($relationType != null) { // Relation type is not null

                $relationEntityRepos = \App::make(GenericRepository::class);
                $relationEntityRepos->setupModel($relatedEntity);

                $relationEntity = $relationEntityRepos->findWithoutFail($relatedEntityId);

                if ($relationType == 'manyToMany') {
                    $entity->{$relatedField}()->attach($relationEntity->id);
                }
            }

            return response()->json([
                'type' => 'success',
                'message' => trans('core::core.entity.created'),
                'action' => 'refresh_datatable'
            ]);
        }
        

        flash(trans($this->languageFile . '.created'))->success();

        return redirect(route($this->routes['index']));
    }

    /**
     * Overwritten function for add image and handle special fields from categories
     *
     * @param $identifier
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function show($identifier) {

        if ($this->permissions['browse'] != '' && !\Auth::user()->hasPermissionTo($this->permissions['browse'])) {
            flash(trans('core::core.you_dont_have_access'))->error();
            return redirect()->route($this->routes['index']);
        }

        $repository = $this->getRepository();

        $entity = $repository->find($identifier);

        if (empty($entity)) {
            flash(trans('core::core.entity.entity_not_found'))->error();
            return redirect(route($this->routes['index']));
        }

        if ($this->blockEntityOwnableAccess()) {
            flash(trans('core::core.you_dont_have_access'))->error();
            return redirect()->route($this->routes['index']);
        }

        $this->entityIdentifier = $entity->id;

        $showView = $this->views['show'];

        $this->entity = $entity;

        $obj = null;

        if($entity->category_id != ''){
            $obj = (object) $entity;

            $newInf = [];

            $category = Category::find($entity->category_id);
            $specialFields = explode(';;',$category->special_fields);
            $assetCategorySpecialFields = explode(';;',$entity->category_special_fields);

            if( $specialFields != '' ) {
                foreach( $this->showFields['information'] as $key => $inf ) {
                    $newInf += [ $key => $inf ];
                    if($key == 'category_id') {
                        foreach( $specialFields as $key => $specialField ) {
                            if( $specialField == '' ){
                                continue;
                            }
                            $newInf += [ $specialField => [ 'type' => 'text' ]];
                            if( isset($assetCategorySpecialFields[$key]) ){

                                $obj->{$specialField} = $assetCategorySpecialFields[$key];
                            }
                        }
                    }
                }
            }
            $this->showFields['information'] = $newInf;
        }

        $this->showFields['information']['age'] =  [ 'type' => 'text' ];
        $this->showFields['information']['current_value'] =  [ 'type' => 'text' ];

        $assetService = \App::make(AssetService::class);
        $entity->age = $assetService->getAge($entity);        
        $entity->current_value = floatval(str_replace(',', '.', str_replace('.', '', $assetService->getCurrentValue($entity))));       
        $entity->purchase_value = floatval(str_replace(',', '.', str_replace('.', '', $entity->purchase_value)));

        $entity->current_value = number_format($assetService->getCurrentValue($entity),2,',','.');        
        $entity->purchase_value = number_format($entity->purchase_value,2,',','.');

        if(isset($entity['useful_life']) && $entity['useful_life'] != ''){
            $entity['useful_life'] = str_replace('.',',',$entity['useful_life']);
            $obj['useful_life'] = str_replace('.',',',$obj['useful_life']);
        }

        $view = view($showView);
        $view->with('entity', $entity->category_id != '' ? $obj : $entity);
        $view->with('show_fields', $this->showFields);
        $view->with('show_fileds_count', count($this->showFields));

        $view->with('next_record', $repository->next($entity));
        $view->with('prev_record', $repository->prev($entity));
        $view->with('disableNextPrev', $this->disableNextPrev);

        $this->setupCustomButtons();
        $this->setupActionButtons();
        $view->with('customShowButtons', $this->customShowButtons);
        $view->with('actionButtons', $this->actionButtons);
        $view->with('commentableExtension', false);
        $view->with('actityLogDatatable', null);
        $view->with('attachmentsExtension', false);
        $view->with('entityIdentifier', $this->entityIdentifier);

        $view->with('hasExtensions', false);

        $view->with('relationTabs', $this->setupRelationTabs($entity));

        $view->with('baseIcons', $this->baseIcons);

        /*
         * Extensions
         */
        if (in_array(self::COMMENTS_EXTENSION, class_uses($this->entity))) {
            $view->with('commentableExtension', true);
            $view->with('hasExtensions', true);
        }
        if (in_array(self::ACTIVITY_LOG_EXTENSION, class_uses($this->entity))) {
            $activityLogDataTable = \App::make(ActivityLogDataTable::class);
            $activityLogDataTable->setEntityData(get_class($entity), $entity->id);
            $view->with('actityLogDatatable', $activityLogDataTable->html());
            $view->with('hasExtensions', true);
        }
        if (in_array(self::ATTACHMENT_EXTENSION, class_uses($this->entity))) {
            $view->with('attachmentsExtension', true);
            $view->with('hasExtensions', true);
        }

        return $view;
    }
    

    /**
     * Overwritten function for add image and handle special fields from categories
     *
     * @param $identifier
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|\Illuminate\View\View
     */
    public function edit($identifier) {

        if ($this->permissions['update'] != '' && !\Auth::user()->hasPermissionTo($this->permissions['update'])) {
            flash(trans('core::core.you_dont_have_access'))->error();
            return redirect()->route($this->routes['index']);
        }

        $repository = $this->getRepository();

        $entity = $repository->find($identifier);

        $this->entity = $entity;


        $obj = null;

        if ( $entity->category_id != '' ) {
            $obj = (object) $entity;

            $newInf = [];

            $category = Category::find($entity->category_id);
            $specialFields = explode(';;',$category->special_fields);
            $assetCategorySpecialFields = explode(';;',$entity->category_special_fields);
            
            if ( $specialFields != '' ) {
                foreach ( $this->showFields['information'] as $key => $inf ){
                    $newInf += [ $key => $inf ];
                    if ( $key == 'category_id' ) {
                        foreach ( $specialFields as $key => $specialField ){
                            if ( $specialField == '' ){
                                continue;
                            }
                            $newInf += [ $specialField => [ 'type' => 'text' ]];
                            if ( isset($assetCategorySpecialFields[$key]) ) {

                                $obj->{$specialField} = $assetCategorySpecialFields[$key];
                            }
                        }
                    }
                }
            }
            $this->showFields['information'] = $newInf;
        }

        if (empty($entity)) {
            flash(trans('core::core.entity.entity_not_found'))->error();

            return redirect(route($this->routes['index']));
        }

        if ($this->blockEntityOwnableAccess()) {
            flash(trans('core::core.you_dont_have_access'))->error();
            return redirect()->route($this->routes['index']);
        }

        $updateForm = $this->form($this->formClass, [
            'method' => 'PATCH',
            'url' => route($this->routes['update'], $entity),
            'id' => 'module_form',
            'class' => 'row',
            'model' => $entity
        ]);

        $updateView = $this->views['edit'];

        $this->entity = $entity;

        $view = view($updateView);
        $view->with('form_request', $this->storeRequest);
        $view->with('entity', $entity->category_id != '' ? $entity : $entity);
        $view->with('show_fields', $this->showFields);
        $view->with('sectionButtons', $this->sectionButtons);


        $view->with('form', $updateForm);

        return $view;
    }

    /**
     * Overwritten function for add image and handle special fields from categories
     *
     * @param $identifier
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function update($identifier) {

        if ($this->permissions['update'] != '' && !\Auth::user()->hasPermissionTo($this->permissions['update'])) {
            flash(trans('core::core.you_dont_have_access'))->error();
            return redirect()->route($this->routes['index']);
        }

        $repository = $this->getRepository();

        $entity = $repository->find($identifier);

        $this->entity = $entity;

        if (empty($entity)) {
            flash(trans('core::core.entity.entity_not_found'))->error();

            return redirect(route($this->routes['index']));
        }

        if ($this->blockEntityOwnableAccess()) {
            flash(trans('core::core.you_dont_have_access'))->error();
            return redirect()->route($this->routes['index']);
        }

        $request = \App::make(Request::class);

        $input = $request->all();

        if(isset($input['purchase_value']) && $input['purchase_value'] != ''){
            $input['purchase_value'] = str_replace('.','',$input['purchase_value']);
        }

        $input['category_special_fields'] = '';

        if ($input['category_id'] != '') {
            $category = Category::find($input['category_id']);
            $specialFields = explode(';;',$category->special_fields);
            if (is_array($specialFields) && count($specialFields) > 0) {
                foreach ($specialFields as $specialField){
                    if ($specialField == '') {
                        continue;
                    }
                    if (isset($input['special_field_'.str_replace(' ','++',$specialField)])) {
                        $input['category_special_fields'] .= $input['special_field_'.str_replace(' ','++',$specialField)].';;';
                    }
                    else {
                        $input['category_special_fields'] .= ';;';
                    }
                }
            }
        }

        $currentOwner = null;
        if ($entity instanceof Ownable && $entity->hasOwner()) {
            $currentOwner = $entity->getOwner();
        }

        $entity = $this->setupAssignedTo($entity, $input);

        $repository = $this->getRepository();

        foreach ($input as $key => $value) {
            if ($value === '') {
                unset($input[$key]);
            }
        }

        if(isset($input['useful_life']) && $input['useful_life'] != ''){
            $input['useful_life'] = str_replace(',','.',$input['useful_life']);
        }

        $entity->update($input);

        $assetImage = $request->file('image_path');

        if ($assetImage != null) {

            \File::delete( UserHelper::ASSET_IMAGE_PATH.$entity->image_path );
    
            $image = 'asset_' . $entity->id . '_.' . $assetImage->getClientOriginalExtension();

            $uploadSuccess = $assetImage->move(UserHelper::ASSET_IMAGE_UPLOAD, $image);

            if ($uploadSuccess) {

                $img = Image::make(base_path() . '/public/' . UserHelper::ASSET_IMAGE_UPLOAD . $image);

                $img->save(base_path() . '/public/' . UserHelper::ASSET_IMAGE_UPLOAD . $image);

                $entity->update([
                    'image_path' => $image
                ]);
            }
        }


        if (config('bap.record_assigned_notification_enabled')) {

            $userService = \App::make(UserService::class);

            $users = $userService->getUsersForNotify(\Auth::user()->company_id); 

            foreach($users as $user){

                try {
                    $commentOn = $entity->asset_number;
                    $commentOn = ' - ' . $commentOn;
                } catch (\Exception $exception) {
                    $commentOn = '';
                }

                $placeholder = new NotificationPlaceholder();
 
                $placeholder->setRecipient(\Auth::user());
                $placeholder->setAuthorUser(\Auth::user());
                $placeholder->setAuthor(\Auth::user()->name);
                $placeholder->setColor('bg-green');
                $placeholder->setIcon('assignment');
                $placeholder->setContent(trans('notifications::notifications.update_record', ['asset' => $entity->asset_number, 'user' => \Auth::user()->name]));

                $placeholder->setUrl(route($this->routes['show'], $entity->id));

                $user->notify(new GenericNotification($placeholder));

            }

        }

        $this->entity = $entity;

        flash(trans("assets::assets.updated"))->success();

        return redirect(route($this->routes['show'], $entity));
    }

}
