<?php

namespace Modules\Suppliers\Datatables;

use Modules\Companies\Entities\Company;
use Yajra\DataTables\EloquentDataTable;
use Modules\Suppliers\Entities\Supplier;
use Modules\Platform\Core\Helper\DataTableHelper;
use Modules\Platform\Core\Datatable\PlatformDataTable;

class SupplierDatatable extends PlatformDataTable {

    const SHOW_URL_ROUTE = 'suppliers.suppliers.show';
    const EDIT_URL_ROUTE = 'suppliers.suppliers.edit';
    const REMOVE_URL_ROUTE = 'suppliers.suppliers.destroy';

    /**
     * Build DataTable class.
     *
     * @param mixed $query Results from query() method.
     * @return \Yajra\DataTables\DataTableAbstract
     */
    public function dataTable($query)
    {
        $dataTable = new EloquentDataTable($query);

        $this->applyLinks($dataTable, self::SHOW_URL_ROUTE, self::EDIT_URL_ROUTE, self::REMOVE_URL_ROUTE);

        return $dataTable;
    }

    /**
     * Get query source of dataTable.
     *
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query(Supplier $model)
    {

        $query = $model->newQuery()
        ->leftJoin('companies', 'suppliers.company_id', '=', 'companies.id')
        ->select([
            'suppliers.id',
            'suppliers.company_id',
            'companies.name as company_name',
            'suppliers.fantasy_name',
            'suppliers.email',
            'suppliers.person_type',
            'suppliers.person_tax_number-company_tax_number',
            'suppliers.responsible_name',
        ]);

        if (!\Auth::user()->hasRole('admin')) {
            $query = $query->where('suppliers.company_id',\Auth::user()->company_id);
        }

        return $query;
    }

    /**
     * Optional method if you want to use html builder.
     *
     * @return \Yajra\DataTables\Html\Builder
     */
    public function html()
    {
        return $this->builder()
            ->columns($this->getColumns())
            ->setTableAttribute('class', 'table table-hover')
            ->parameters([
                'dom' => "<'row'<'col-sm-12'tr>>\n\t\t\t<'row'<'col-sm-12 col-md-5'i><'col-sm-12 col-md-7 dataTables_pager'lp>>",
                'stateSave' => true,
                'columnFilters' => $this->getFilters(),
                'buttons' => DataTableHelper::buttons(),
                'regexp' => true
            ]);
    }

    /**
     * @return array
     */
    protected function getFilters() {


        $currentColumn = 0;
        $columnFilters = [];

        if(\Auth::user()->hasRole('admin')){

            $data_companies = [];
            foreach (Company::all()->toArray() as $company){
                $data_companies[] = [
                    'value' => $company['id'],
                    'label' => $company['name'],
                ];
            }

            $columnFilters[] = [
                    'column_number' => $currentColumn++,
                    'filter_type' => 'select',
                    'select_type' => 'select2',
                    'select_type_options' => [
                        'theme' => "bootstrap",
                        'width' => '100%'
                    ],
                    'data' => $data_companies                    
                ];
        }
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];

        return $columnFilters;
    }

    /**
     * @return array
     */
    protected function getColumns() {

        $data = [];

        if(\Auth::user()->hasRole('admin')){
            $data += [
                'company_name' => [
                    'name' => 'company_id',
                    'data' => 'company_name',
                    'title' => trans('suppliers::suppliers.table.company_name'),
                    'data_type' => 'text'
                ],
            ];
        }
        $data += [
                'fantasy_name' => [
                'data' => 'fantasy_name',
                'title' => trans('suppliers::suppliers.table.fantasy_name'),
                'data_type' => 'text'
            ],
            'email' => [
                'data' => 'email',
                'title' => trans('suppliers::suppliers.table.email'),
                'data_type' => 'text'
            ],
            'person_type' => [
                'data' => 'person_type',
                'title' => trans('suppliers::suppliers.table.person_type'),
                'data_type' => 'text'
            ],
            'person_tax_number-company_tax_number' => [
                'data' => 'person_tax_number-company_tax_number',
                'title' => trans('suppliers::suppliers.table.person_tax_number-company_tax_number'),
                'data_type' => 'text'
            ],
            'responsible_name' => [
                'data' => 'responsible_name',
                'title' => trans('suppliers::suppliers.table.responsible_name'),
                'data_type' => 'text'
            ],
            'actions' => ['title' => trans('core::core.table.actions'), 'data_type' => 'actions', 'orderable' => false],
        ];

        return $data;

    }

}
