<?php

namespace Modules\Reports\Datatables;

use Modules\Assets\Entities\Asset;
use Modules\Places\Entities\Place;
use Modules\Assets\Entities\Status;
use Modules\Assets\Entities\Category;
use Modules\Companies\Entities\Company;
use Yajra\DataTables\EloquentDataTable;
use Modules\Collaborators\Entities\Area;
use Modules\Collaborators\Entities\Collaborator;
use Modules\Platform\Core\Helper\DataTableHelper;
use Modules\Platform\Core\Datatable\PlatformDataTable; 

class ReportDatatable extends PlatformDataTable {


    const QUERY_BY_PLACE   = 1;
    const QUERY_BY_COMPANY = 2;
    const GENERAL_QUERY    = 3;

    const SHOW_URL_ROUTE = 'assets.assets.show';
    const EDIT_URL_ROUTE = 'assets.assets.edit';
    const REMOVE_URL_ROUTE = 'assets.assets.destroy';

    protected $idQuery;
    protected $queryType;

    /**
     * Build DataTable class.
     *
     * @param mixed $query Results from query() method.
     * @return \Yajra\DataTables\DataTableAbstract
     */
    public function dataTable($query)
    {
        $dataTable = new EloquentDataTable($query);

        $this->applyLinks($dataTable, self::SHOW_URL_ROUTE, self::EDIT_URL_ROUTE, self::REMOVE_URL_ROUTE);

        return $dataTable;
    }

    /**
     * Get query source of dataTable.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query(Asset $model)
    {

        $query = $model->newQuery()
        ->leftJoin('status', 'assets.status_id', '=', 'status.id')
        ->leftJoin('areas', 'assets.area_id', '=', 'areas.id')
        ->leftJoin('categories', 'assets.category_id', '=', 'categories.id')
        ->select([
            'assets.id',
            'assets.company_id',
            'assets.status_id',
            'assets.area_id',
            'assets.category_id',
            'assets.place_id',
            'assets.asset_number',
            'areas.name as area_name',
            'status.name as status_name',
            'categories.name as category_name',
        ]);

        if($this->queryType == $this::QUERY_BY_PLACE){
            $query = $query->where('assets.place_id',$this->idQuery);
        }
        if($this->queryType == $this::QUERY_BY_COMPANY){
            $query = $query->where('assets.company_id',$this->idQuery);
        }

        return $query;
    }

    /**
     * Optional method if you want to use html builder.
     *
     * @return \Yajra\DataTables\Html\Builder
     */
    public function html()
    {
        $currentColumn = 0;
        $columnFilters = [];

        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];

        $data_areas = [];

        $areas = Area::all()->toArray();

        foreach ($areas as $area){
            $data_areas[] = [
                'value' => $area['id'],
                'label' => $area['name'],
            ];
        }

        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'select',
            'select_type' => 'select2',
            'select_type_options' => [
                'theme' => "bootstrap",
                'width' => '100%'
            ],
            'data' => $data_areas                    
        ];

        $data_status = [];

        $status = Status::all()->toArray();

        foreach ($status as $status) {
            $data_status[] = [
                'value' => $status['id'],
                'label' => $status['name'],
            ];
        }

        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'select',
            'select_type' => 'select2',
            'select_type_options' => [
                'theme' => "bootstrap",
                'width' => '100%'
            ],
            'data' => $data_status                    
        ];

        $data_categories = [];

        $categories = Category::all()->toArray();

        foreach ($categories as $category){
            $data_categories[] = [
                'value' => $category['id'],
                'label' => $category['name'],
            ];
        }

        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'select',
            'select_type' => 'select2',
            'select_type_options' => [
                'theme' => "bootstrap",
                'width' => '100%'
            ],
            'data' => $data_categories                    
        ];


        return $this->builder()
            ->columns($this->getColumns())

            ->setTableAttribute('class', 'table responsive table-hover')
            ->parameters([
                'dom' => "<'row'<'col-sm-12'tr>>\n\t\t\t<'row'<'col-sm-12 col-md-5'i><'col-sm-12 col-md-7 dataTables_pager'lp>>",

                'stateSave' => true,
                'columnFilters' => $columnFilters,
                'buttons' => DataTableHelper::buttons(),
                'regexp' => true,

        ]);
    }

    /**
     * @return array
     */
    protected function getColumns() {

        $data = [];

        $data += [
            'asset_number' => [
                'data' => 'asset_number',
                'title' => trans('reports::reports.table.asset_number'),
                'data_type' => 'text'
            ],
            'area_name' => [
                'name' => 'area_id',
                'data' => 'area_name',
                'title' => trans('reports::reports.table.area_name'),
                'data_type' => 'text'
            ],
            'status_name' => [
                'name' => 'status_id',
                'data' => 'status_name',
                'title' => trans('reports::reports.table.status_name'),
                'data_type' => 'text'
            ],
            'category_name' => [
                'name' => 'category_id',
                'data' => 'category_name',
                'title' => trans('reports::reports.table.category_name'),
                'data_type' => 'text'
            ],
        ];

        return $data;

    }

    public function setIdQuery($id){
        $this->idQuery = $id;
    }

    public function setQueryType($type){
        $this->queryType = $type;
    }

}
