<?php

namespace Modules\Platform\MenuManager\Repositories;

use Carbon\Carbon;
use Illuminate\Support\Facades\Lang;
use Modules\Platform\Core\Repositories\PlatformRepository;
use Modules\Platform\MenuManager\Helper\MenuHelper;
use Spatie\Menu\Laravel\Facades\Menu;
use Spatie\Menu\Laravel\Html;
use Spatie\Menu\Laravel\View as SpatieView;
use Spatie\Menu\ActiveUrlChecker;

use View;

/**
 * Menu Repo
 *
 * Class MenuRepository
 * @package Modules\Platform\MenuManager\Repositories
 */
class MenuRepository extends PlatformRepository
{

    /**
     * @return string
     */
    public function model()
    {
        return \Modules\Platform\MenuManager\Entities\Menu::class;
    }

    /**
     * Render Main menu
     *
     * @return static
     */
    public function renderMainMenu()
    {
        if (\Cache::has(MenuHelper::MAIN_MENU_CACHE_KEY)) {
            $mainMenuEloquent = \Cache::get(MenuHelper::MAIN_MENU_CACHE_KEY);
        } else {
            $mainMenuEloquent = $this->getMainMenu();
            \Cache::put(MenuHelper::MAIN_MENU_CACHE_KEY, $mainMenuEloquent, Carbon::now()->addDay(1));
        }


        $html = '<ul class="m-menu__nav  m-menu__nav--submenu-arrow ">';



        foreach ($mainMenuEloquent as $menuElement) {
            if ($menuElement->parent_id == null) {
                $item = $this->renderElement($menuElement);
                $html .= $item;
            }
        }

        $html .= '</ul>';
        

        return $html;
    }

    /**
     * Get Main Menu form Database
     *
     * @return mixed
     */
    public function getMainMenu()
    {
        $result = $this->orderBy('order_by', 'asc')
            ->findWhere([
                'section' => MenuHelper::MAIN_MENU
            ]);

        return $result;
    }

    private function renderSubMenu($menuElement, $parentLevel = 0)
    {

        

        $html  = '<div class="m-menu__submenu m-menu__submenu--classic m-menu__submenu--' . ( ($parentLevel > 0) ? 'right' : 'left' ) . '">';
        $html .= '<span class="m-menu__arrow m-menu__arrow--adjust"></span>';
        $html .= '<ul class="m-menu__subnav">';

        foreach ($menuElement->children as $element) {
            $item = $this->renderElement($element, $parentLevel + 1);
            $html .= $item;
        }

        $html .= '</ul>';
        $html .= '</div>';

        if ($menuElement->visibility > 0) {
            return $html;
        }
    }

     /**
     * Recursive menu render
     * @param $mainMenu
     * @param $menuElement
     */
    private function renderElement($menuElement, $parentLevel = 0)
    {
        if($menuElement->permission != '' && ! \Auth::user()->hasPermissionTo($menuElement->permission)){
            return;
        }

        $langPrefix = 'core::core.menu.';
        $url = \Request::path();

        if (Lang::has('bap_menu.home')) {
            $langPrefix = 'bap_menu.';
        }

        $item = '';

        $isActive = ActiveUrlChecker::check($menuElement->url, $url);

        if ($menuElement->children->count() > 0) {

            $item .= '<li class="m-menu__item  m-menu__item--submenu m-menu__item--rel' . ( $isActive ? ' m-menu__item--active' : '' ) .'" m-menu-submenu-toggle="click" aria-haspopup="true">';

            $view = View::make('core::menu-element', [
                'icon' => $menuElement->icon,
                'cssClass' => 'm-menu__toggle',
                'dropdown' => true,
                'name' => $menuElement->dont_translate == true ? $menuElement->label : Lang::get($langPrefix . $menuElement->label),
                'url' => $menuElement->url
            ]);

            $item .= $view->render();

            $item .= $this->renderSubMenu($menuElement, $parentLevel);

            $item .= '</li>';

        } else {

            $item = '<li class="m-menu__item'. ( $isActive ? ' m-menu__item--active' : '' ) .'">';

            if ($menuElement->permission != '' && \Auth::user()->hasPermissionTo($menuElement->permission)) {

                $view = View::make('core::menu-element', [
                    'icon' => $menuElement->icon,
                    'name' => $menuElement->dont_translate == true ? $menuElement->label : Lang::get($langPrefix . $menuElement->label),
                    'url' => $menuElement->url
                ]);

                $item .= $view->render();

            } else if ($menuElement->visibility > 0) {

                $view = View::make('core::menu-element', [
                    'icon' => $menuElement->icon,
                    'name' => $menuElement->dont_translate == true ? $menuElement->label : Lang::get($langPrefix . $menuElement->label),
                    'url' => $menuElement->url
                ]);

                $item .= $view->render();
            }

            $item .= '</li>';

        }

        return $item;
    }

    /**
     * Recursive menu render
     * @param $mainMenu
     * @param $menuElement
     */
    private function renderMenuElement($mainMenu, $menuElement, $parentLevel = 0)
    {
        $langPrefix = 'core::core.menu.';

        if (Lang::has('bap_menu.home')) {
            $langPrefix = 'bap_menu.';
        }

        if ($menuElement->children->count() > 0) {

            $submenu = Menu::new();

            if ($parentLevel == 1) {

            }

            $submenu->addClass('m-menu__subnav');

            foreach ($menuElement->children as $element) {
                $this->renderMenuElement($submenu, $element, $parentLevel + 1);
            }



            if ($menuElement->visibility > 0) {
                $mainMenu->submenu(
                    SpatieView::create('core::menu-element', [
                        'cssClass' => 'menu-toggle',
                        'icon' => $menuElement->icon,
                        'name' => $menuElement->dont_translate == true ? $menuElement->label : Lang::get($langPrefix. $menuElement->label),
                        'url' => 'javascript:void(0);'
                    ]),
                    $submenu
                );
            }

        } else {

            if ($menuElement->permission != '') {
                
                $mainMenu->addIf(\Auth::user()->hasPermissionTo($menuElement->permission), SpatieView::create('core::menu-element', [
                    'icon' => $menuElement->icon,
                    'name' => $menuElement->dont_translate == true ? $menuElement->label : Lang::get($langPrefix . $menuElement->label),
                    'url' => $menuElement->url
                ])->addParentClass('m-menu__item'));

            } else {
                
                if ($menuElement->visibility > 0) {

                    $mainMenu->add(SpatieView::create('core::menu-element', [
                        'icon' => $menuElement->icon,
                        'name' => $menuElement->dont_translate == true ? $menuElement->label : Lang::get($langPrefix . $menuElement->label),
                        'url' => $menuElement->url
                    ])->addParentClass('m-menu__item'));

                }

            }
        }

    }
}
