<?php

namespace Modules\Platform\Core\Datatable;

use Modules\Platform\Core\Helper\DataTableHelper;
use Spatie\Activitylog\Models\Activity;
use Yajra\DataTables\EloquentDataTable;

/**
 * Class ActivityLogDataTable
 * @package Modules\Platform\Core\Datatable
 */
class ActivityLogDataTable extends PlatformDataTable
{

    /** 
     * @var
     */
    private $entityId;

    /**
     * @var
     */
    private $entityClass;

    /**
     * @param $entityClass
     * @param $entityId
     */
    public function setEntityData($entityClass, $entityId)
    {
        $this->entityClass = $entityClass;
        $this->entityId = $entityId;
    }


    /**
     * Build DataTable class.
     *
     * @param mixed $query Results from query() method.
     * @return \Yajra\DataTables\DataTableAbstract
     */
    public function dataTable($query)
    {
        $dataTable = new EloquentDataTable($query);

        $this->applyLinks($dataTable, 'core.activity-log-details');

        $dataTable->addRowAttr('class', function ($record) {
            return 'open-in-modal';
        });


        $dataTable->filterColumn('updated_at', function ($query, $keyword) {
            $dates = DataTableHelper::getDatesForFilter($keyword);

            if ($dates != null) {
                $query->whereBetween('updated_at', array($dates[0], $dates[1]));
            }
        });

        return $dataTable;
    }

    /**
     * Get query source of dataTable.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query(Activity $model)
    {

        return $model->newQuery()->with('causer')
        ->leftJoin('users', 'activity_log.causer_id', '=', 'users.id')
        ->leftJoin('assets', 'activity_log.subject_id', '=', 'assets.id')
        ->leftJoin('places', 'assets.place_id', '=', 'places.id')
        ->leftJoin('areas', 'assets.area_id', '=', 'areas.id')
        ->leftJoin('status', 'assets.status_id', '=', 'status.id')
        ->select([
            'activity_log.id',
            'activity_log.log_name',
            'activity_log.description',
            'activity_log.subject_id',
            'activity_log.subject_type',
            'activity_log.causer_id',
            'activity_log.causer_type',
            'activity_log.properties',
            'activity_log.created_at',
            'activity_log.updated_at',
            'users.name as user_name',
            'places.identifier_name as asset_place',
            'areas.name as asset_area',
            'status.name as asset_status',
        ]);

    }


    /**
     * Optional method if you want to use html builder.
     *
     * @return \Yajra\DataTables\Html\Builder
     */
    public function html()
    {
        return $this->builder()
            ->setTableId('ActivityLogDatatable')
            ->columns($this->getColumns())
            ->minifiedAjax(route('core.activity-log', ['model' => $this->entityClass,'entityId' =>$this->entityId]))
            ->setTableAttribute('class', 'table table-hover')
            ->parameters([
                'dom' => "<'row'<'col-sm-12'tr>>\n\t\t\t<'row'<'col-sm-12 col-md-5'i><'col-sm-12 col-md-7 dataTables_pager'lp>>",

                'stateSave' => false,
                'order' => [2, 'desc'],
                'columnFilters' => [
                    [
                        'column_number' => 0,
                        'filter_type' => 'bap_date_range_picker'
                    ],
                    [
                        'column_number' => 1,
                        'filter_type' => 'text'
                    ],
                    [
                        'column_number' => 2,
                        'filter_type' => 'text'
                    ],
                    [
                        'column_number' => 3,
                        'filter_type' => 'text'
                    ],
                    [
                        'column_number' => 4,
                        'filter_type' => 'text'
                    ],
                    [
                        'column_number' => 5,
                        'filter_type' => 'text'
                    ],
                ],
                'buttons' => DataTableHelper::buttons(),
                'regexp' => true

            ]);
    }

    /**
     * @return array
     */
    protected function getColumns()
    {

        return
            [
                'updated_at' => [
                    'data' => 'updated_at',
                    'title' => trans('assets::assets.activity_table.updated_at'),
                    'data_type' => 'datetime'
                ],
                'description' => [
                    'data' => 'description',
                    'title' => trans('assets::assets.activity_table.description'),
                    'data_type' => 'text'
                ],
                'user_name' => [
                    'data' => 'user_name',
                    'title' => trans('assets::assets.activity_table.updated_by'),
                    'data_type' => 'text',

                ],
                'asset_place' => [
                    'data' => 'asset_place',
                    'title' => trans('assets::assets.activity_table.place'),
                    'data_type' => 'text',

                ],
                'asset_area' => [
                    'data' => 'asset_area',
                    'title' => trans('assets::assets.activity_table.area'),
                    'data_type' => 'text',

                ],
                'asset_status' => [
                    'data' => 'asset_status',
                    'title' => trans('assets::assets.activity_table.status'),
                    'data_type' => 'text',

                ],
            ];
    }
}
