<?php

namespace Modules\Places\Datatables;

 
use Modules\Places\Entities\Place;
use Modules\Companies\Entities\Company;
use Yajra\DataTables\EloquentDataTable;
use Modules\Platform\Core\Helper\DataTableHelper;
use Modules\Platform\Core\Datatable\PlatformDataTable;

class PlaceDatatable extends PlatformDataTable {

    const SHOW_URL_ROUTE = 'places.places.show';
    const EDIT_URL_ROUTE = 'places.places.edit';
    const REMOVE_URL_ROUTE = 'places.places.destroy';

    /**
     * Build DataTable class.
     *
     * @param mixed $query Results from query() method.
     * @return \Yajra\DataTables\DataTableAbstract
     */
    public function dataTable($query) {

        $dataTable = new EloquentDataTable($query);

        $this->applyLinks($dataTable, self::SHOW_URL_ROUTE, self::EDIT_URL_ROUTE, self::REMOVE_URL_ROUTE);

        return $dataTable;
    }

    /**
     * Get query source of dataTable.
     *
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query(Place $model)
    {
        $query = $model->newQuery()
        ->leftJoin('companies', 'places.company_id', '=', 'companies.id')
        ->select([
            'companies.name as company_name',
            'places.id',
            'places.company_id',
            'places.identifier_name',
            'places.state',
            'places.city',
        ]);

        if (\Auth::user()->hasRole('chefe')) {
            $query = $query->where('places.company_id',\Auth::user()->company_id);
        }
        elseif(\Auth::user()->hasRole('gerente')){
            return null;
        }

        return $query;

    }

    /**
     * Optional method if you want to use html builder.
     *
     * @return \Yajra\DataTables\Html\Builder
     */
    public function html()
    {
        return $this->builder()
            ->columns($this->getColumns())

            ->setTableAttribute('class', 'table table-hover')
            ->parameters([
                'dom' => "<'row'<'col-sm-12'tr>>\n\t\t\t<'row'<'col-sm-12 col-md-5'i><'col-sm-12 col-md-7 dataTables_pager'lp>>",
                'stateSave' => true,
                'columnFilters' => $this->getFilters(),
                'buttons' => DataTableHelper::buttons(),
                'regexp' => true
            ]);
    }

    /**
     * @return array
     */
    protected function getFilters(){


        $currentColumn = 0;
        $columnFilters = [];

        if(\Auth::user()->hasRole('admin')){

            $data_companies = [];
            foreach (Company::all()->toArray() as $company){
                $data_companies[] = [
                    'value' => $company['id'],
                    'label' => $company['name'],
                ];
            }

            $columnFilters[] = [
                    'column_number' => $currentColumn++,
                    'filter_type' => 'select',
                    'select_type' => 'select2',
                    'select_type_options' => [
                        'theme' => "bootstrap",
                        'width' => '100%'
                    ],
                    'data' => $data_companies                    
                ];
        }

        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];

        return $columnFilters;
    }

    /**
     * @return array
     */
    protected function getColumns()
    {

        $data = [];

        if(\Auth::user()->hasRole('admin')){
            $data += [
                'company_name' => [
                    'name' => 'company_id',
                    'data' => 'company_name',
                    'title' => trans('places::places.table.company_name'),
                    'data_type' => 'text'
                ],
            ];
        }
        $data += [

            'identifier_name' => [
                'data' => 'identifier_name',
                'title' => trans('places::places.table.identifier_name'),
                'data_type' => 'text'
            ],
            'state' => [
                'data' => 'state',
                'title' => trans('places::places.table.state'),
                'data_type' => 'text'
            ],
            'city' => [
                'data' => 'city',
                'title' => trans('places::places.table.city'),
                'data_type' => 'text'
            ],
            'actions' => ['title' => trans('core::core.table.actions'), 'data_type' => 'actions', 'orderable' => false],
        ];

        return $data;
    }

}
