<?php

namespace Modules\Dashboard\Datatables;

use Modules\Assets\Entities\Asset;
use Modules\Places\Entities\Place;
use Modules\Assets\Entities\Status;
use Modules\Companies\Entities\Company;
use Yajra\DataTables\EloquentDataTable;
use Modules\Collaborators\Entities\Area;
use Modules\Collaborators\Entities\Collaborator;
use Modules\Platform\Core\Helper\DataTableHelper;
use Modules\Platform\Core\Datatable\PlatformDataTable;

class DashboardDatatable extends PlatformDataTable
{

    const SHOW_URL_ROUTE = 'assets.assets.show';
    const EDIT_URL_ROUTE = 'assets.assets.edit';
    const REMOVE_URL_ROUTE = 'assets.assets.destroy';

    /**
     * Build DataTable class.
     *
     * @param mixed $query Results from query() method.
     * @return \Yajra\DataTables\DataTableAbstract
     */
    public function dataTable($query)
    {
        $dataTable = new EloquentDataTable($query);
 
        $this->applyLinks($dataTable, self::SHOW_URL_ROUTE, self::EDIT_URL_ROUTE, self::REMOVE_URL_ROUTE);

        $dataTable->filterColumn('created_at', function ($query, $keyword) {

            $dates = DataTableHelper::getDatesForFilter($keyword);

            if ($dates != null) {
                $query->whereBetween('created_at', array($dates[0], $dates[1]));
            }
        });

        $dataTable->filterColumn('updated_at', function ($query, $keyword) {

            $dates = DataTableHelper::getDatesForFilter($keyword);

            if ($dates != null) {
                $query->whereBetween('updated_at', array($dates[0], $dates[1]));
            }
        });

        return $dataTable;
    }

    /**
     * Get query source of dataTable.
     *
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query(Asset $model) {

        $query = $model->newQuery()
        ->leftJoin('status', 'assets.status_id', '=', 'status.id')
        ->leftJoin('areas', 'assets.area_id', '=', 'areas.id')
        ->leftJoin('places', 'assets.place_id', '=', 'places.id')
        ->leftJoin('companies', 'assets.company_id', '=', 'companies.id')
        ->select([
            'assets.id',
            'assets.company_id',
            'assets.status_id',
            'assets.area_id',
            'assets.category_id',
            'assets.place_id',
            'assets.asset_number',
            'areas.name as area_name',
            'status.name as status_name',
            'places.identifier_name as place_name',
            'companies.name as company_name',
        ]);

        if (\Auth::user()->hasRole('chefe')) {
            $query = $query->where('assets.company_id',\Auth::user()->company_id);
        }
        else if(\Auth::user()->hasRole('gerente')){
            $query = $query->where('assets.place_id',\Auth::user()->place_id);
        }

        return $query;
    }

    /**
     * Optional method if you want to use html builder.
     *
     * @return \Yajra\DataTables\Html\Builder
     */
    public function html() {

        return $this->builder()
            ->columns($this->getColumns())
            ->setTableAttribute('class', 'table responsive table-hover')
            ->parameters([
                'dom' => "<'row'<'col-sm-12'tr>>\n\t\t\t<'row'<'col-sm-12 col-md-5'i><'col-sm-12 col-md-7 dataTables_pager'lp>>",
                'stateSave' => true,
                'columnFilters' => $this->getFilters(),
                'buttons' => DataTableHelper::buttons(),
                'regexp' => true

            ]);
    }

    /**
     * @return array
     */
    protected function getFilters() {


        $currentColumn = 0;
        $columnFilters = [];

        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];        

        if(\Auth::user()->hasRole('admin')){

            $data_companies = [];
            foreach (Company::all()->toArray() as $company){
                $data_companies[] = [
                    'value' => $company['id'],
                    'label' => $company['name'],
                ];
            }

            $columnFilters[] = [
                    'column_number' => $currentColumn++,
                    'filter_type' => 'select',
                    'select_type' => 'select2',
                    'select_type_options' => [
                        'theme' => "bootstrap",
                        'width' => '100%'
                    ],
                    'data' => $data_companies                    
                ];
        }
        if(\Auth::user()->hasRole('admin') || \Auth::user()->hasRole('chefe')){

            $places = [];
            $data_places = [];
            if (\Auth::user()->hasRole('admin')) {
                $places = Place::all()->toArray();
            }
            else {
                $places = Place::where('company_id', \Auth::user()->company_id)->get()->toArray();
            }
            foreach ($places as $place){
                $data_places[] = [
                    'value' => $place['id'],
                    'label' => $place['identifier_name'],
                ];
            }

            $columnFilters[] = [
                'column_number' => $currentColumn++,
                'filter_type' => 'select',
                'select_type' => 'select2',
                'select_type_options' => [
                    'theme' => "bootstrap",
                    'width' => '100%'
                ],
                'data' => $data_places                    
            ];
        }


        $areas = [];
        $data_areas = [];

        $areas = Area::all()->toArray();

        foreach ($areas as $area){
            $data_areas[] = [
                'value' => $area['id'],
                'label' => $area['name'],
            ];
        }

        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'select',
            'select_type' => 'select2',
            'select_type_options' => [
                'theme' => "bootstrap",
                'width' => '100%'
            ],
            'data' => $data_areas                    
        ];

        $status = [];
        $data_status = [];

        $status = Status::all()->toArray();
    
        foreach ($status as $status){
            $data_status[] = [
                'value' => $status['id'],
                'label' => $status['name'],
            ];
        }

        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'select',
            'select_type' => 'select2',
            'select_type_options' => [
                'theme' => "bootstrap",
                'width' => '100%'
            ],
            'data' => $data_status                    
        ];

        return $columnFilters;
    }

    /**
     * @return array
     */
    protected function getColumns() {

        $data = [];

        $data += [
            'asset_number' => [
                'data' => 'asset_number',
                'title' => trans('dashboard::dashboard.table.asset_number'),
                'data_type' => 'text'
            ],
        ];

        if(\Auth::user()->hasRole('admin')){
            $data += [
                'company_name' => [
                    'name' => 'company_id',
                    'data' => 'company_name',
                    'title' => trans('dashboard::dashboard.table.company_name'),
                    'data_type' => 'text'
                ],
            ];
        }
        if(\Auth::user()->hasRole('admin') || \Auth::user()->hasRole('chefe')){
            $data += [
                'place_name' => [
                    'name' => 'place_id',
                    'data' => 'place_name',
                    'title' => trans('dashboard::dashboard.table.place_name'),
                    'data_type' => 'text'
                ],
            ];
        }

        $data += [

            'area_name' => [
                'name' => 'area_id',
                'data' => 'area_name',
                'title' => trans('dashboard::dashboard.table.area_name'),
                'data_type' => 'text'
            ],
            'status_name' => [
                'name' => 'status_id',
                'data' => 'status_name',
                'title' => trans('dashboard::dashboard.table.status_name'),
                'data_type' => 'text'
            ],
        ];

        return $data;

    }

}

