<?php

namespace Modules\Collaborators\Datatables;


use Modules\Places\Entities\Place;
use Modules\Companies\Entities\Company;
use Yajra\DataTables\EloquentDataTable;
use Modules\Collaborators\Entities\Area;
use Modules\Collaborators\Entities\Collaborator;
use Modules\Platform\Core\Helper\DataTableHelper;
use Modules\Platform\Core\Datatable\PlatformDataTable;

class CollaboratorDatatable extends PlatformDataTable
{

    const SHOW_URL_ROUTE = 'collaborators.collaborators.show';
    const EDIT_URL_ROUTE = 'collaborators.collaborators.edit';
    const REMOVE_URL_ROUTE = 'collaborators.collaborators.destroy';

    /**
     * Build DataTable class.
     *
     * @param mixed $query Results from query() method.
     * @return \Yajra\DataTables\DataTableAbstract
     */
    public function dataTable($query)
    {
        $dataTable = new EloquentDataTable($query);

        $this->applyLinks($dataTable, self::SHOW_URL_ROUTE, self::EDIT_URL_ROUTE, self::REMOVE_URL_ROUTE);

        $dataTable->filterColumn('created_at', function ($query, $keyword) {

            $dates = DataTableHelper::getDatesForFilter($keyword);

            if ($dates != null) {
                $query->whereBetween('created_at', array($dates[0], $dates[1]));
            }
        });
        $dataTable->filterColumn('updated_at', function ($query, $keyword) {

            $dates = DataTableHelper::getDatesForFilter($keyword);

            if ($dates != null) {
                $query->whereBetween('updated_at', array($dates[0], $dates[1]));
            }
        });


        return $dataTable;
    }

    /**
     * Get query source of dataTable.
     *
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query(Collaborator $model)
    {
        $query = $model->newQuery()
        ->leftJoin('companies', 'collaborators.company_id', '=', 'companies.id')
        ->leftJoin('places', 'collaborators.place_id', '=', 'places.id')
        ->leftJoin('areas', 'collaborators.area_id', '=', 'areas.id')
        ->select([
            'collaborators.id',
            'collaborators.company_id',
            'collaborators.place_id',
            'collaborators.area_id',
            'collaborators.name',
            'collaborators.email',
            'collaborators.phone',
            'collaborators.mobile',
            'collaborators.city',
            'collaborators.state',
            'collaborators.person_tax_number',
            'places.identifier_name as place_name',
            'companies.name as company_name',
            'areas.name as area_name',
        ]);

        if (\Auth::user()->hasRole('chefe')) {
            $query = $query->where('collaborators.company_id',\Auth::user()->company_id);
        }
        else if(\Auth::user()->hasRole('gerente')){
            $query = $query->where('collaborators.place_id',\Auth::user()->place_id);
        }

        return $query;
    }

    /**
     * Optional method if you want to use html builder.
     *
     * @return \Yajra\DataTables\Html\Builder
     */
    public function html()
    {
        return $this->builder()
            ->columns($this->getColumns())

            ->setTableAttribute('class', 'table table-hover')
            ->parameters([
                'dom' => "<'row'<'col-sm-12'tr>>\n\t\t\t<'row'<'col-sm-12 col-md-5'i><'col-sm-12 col-md-7 dataTables_pager'lp>>",

                'stateSave' => true,
                'columnFilters' => $this->getFilters(),
                'buttons' => DataTableHelper::buttons(),
                'regexp' => true

            ]);
    }

    /**
     * @return array
     */
    protected function getFilters(){

        $currentColumn = 0;
        $columnFilters = [];

        if(\Auth::user()->hasRole('admin')){

            $data_companies = [];
            foreach (Company::all()->toArray() as $company){
                $data_companies[] = [
                    'value' => $company['id'],
                    'label' => $company['name'],
                ];
            }

            $columnFilters[] = [
                    'column_number' => $currentColumn++,
                    'filter_type' => 'select',
                    'select_type' => 'select2',
                    'select_type_options' => [
                        'theme' => "bootstrap",
                        'width' => '100%'
                    ],
                    'data' => $data_companies                    
                ];
        }
        if(\Auth::user()->hasRole('admin') || \Auth::user()->hasRole('chefe')){

            $places = [];
            $data_places = [];
            if (\Auth::user()->hasRole('admin')) {
                $places = Place::all()->toArray();
            }
            else {
                $places = Place::where('company_id', \Auth::user()->company_id)->get()->toArray();
            }
            foreach ($places as $place){
                $data_places[] = [
                    'value' => $place['id'],
                    'label' => $place['identifier_name'],
                ];
            }

            $columnFilters[] = [
                'column_number' => $currentColumn++,
                'filter_type' => 'select',
                'select_type' => 'select2',
                'select_type_options' => [
                    'theme' => "bootstrap",
                    'width' => '100%'
                ],
                'data' => $data_places                    
            ];
        }


        $data_areas = [];

        $areas = Area::all()->toArray();
    
        foreach ($areas as $area){
            $data_areas[] = [
                'value' => $area['id'],
                'label' => $area['name'],
            ];
        }

        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'select',
            'select_type' => 'select2',
            'select_type_options' => [
                'theme' => "bootstrap",
                'width' => '100%'
            ],
            'data' => $data_areas                    
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];
        $columnFilters[] = [
            'column_number' => $currentColumn++,
            'filter_type' => 'text'
        ];

        return $columnFilters;
    }

    /**
     * @return array
     */
    protected function getColumns()
    {

        $data = [];

        if(\Auth::user()->hasRole('admin')){
            $data += [
                'company_name' => [
                    'name' => 'company_id',
                    'data' => 'company_name',
                    'title' => trans('collaborators::collaborators.table.company_name'),
                    'data_type' => 'text'
                ],
            ];
        }
        if(\Auth::user()->hasRole('admin') || \Auth::user()->hasRole('chefe')){
            $data += [
                'place_name' => [
                    'name' => 'place_id',
                    'data' => 'place_name',
                    'title' => trans('collaborators::collaborators.table.place_name'),
                    'data_type' => 'text'
                ],
            ];
        }

        $data += [

            'area_name' => [
                'name' => 'area_id',
                'data' => 'area_name',
                'title' => trans('collaborators::collaborators.table.area_name'),
                'data_type' => 'text'
            ],
            'name' => [
                'data' => 'name',
                'title' => trans('collaborators::collaborators.table.name'),
                'data_type' => 'text'
            ],
            'email' => [
                'data' => 'email',
                'title' => trans('collaborators::collaborators.table.email'),
                'data_type' => 'text'
            ],
            'phone' => [
                'data' => 'phone',
                'title' => trans('collaborators::collaborators.table.phone'),
                'data_type' => 'text'
            ],
            'mobile' => [
                'data' => 'mobile',
                'title' => trans('collaborators::collaborators.table.mobile'),
                'data_type' => 'text'
            ],
            'city' => [
                'data' => 'city',
                'title' => trans('collaborators::collaborators.table.city'),
                'data_type' => 'text'
            ],
            'state' => [
                'data' => 'state',
                'title' => trans('collaborators::collaborators.table.state'),
                'data_type' => 'text'
            ],
            'cpf' => [
                'data' => 'cpf',
                'title' => trans('collaborators::collaborators.table.cpf'),
                'data_type' => 'text'
            ],
            'actions' => ['title' => trans('core::core.table.actions'), 'data_type' => 'actions', 'orderable' => false],
        ];

        return $data;

    }

}
